# encoding=utf-8
__author__ = "Roland Trouville"
__copyright__ = "Copyright 2015+, Consortium MonPaGe"
__license__ = "Creative Commons 4.0 By-Nc-Sa"
__maintainer__ = "Roland Trouville"
__email__ = "contact.monpage@gmail.com"
__status__ = "Production"

from .cotation_acoustic_analysis import CotationAcousticAnalysis

try:
	from typing import override
except ImportError:
	from typing_extensions import override  # noqa: F401

from typing import Union
from cotation.acoustic.display.io_widget.output import Output
from cotation.acoustic.display.io_widget.info import Info
from cotation.acoustic.display.io_widget.info import InfoBox

# @property transparantly have getter and setter.
# class.x instead of class.(get,set)_x()

DEFAULT_SOUND_FILE_PATTERN = "{}_{}_Module{}.wav"


class Atenu(CotationAcousticAnalysis):
	"""
	Handles the acoustic analysis of a sustained /a/ sound (a tenu) in a phonatory task.

	This class defines interactive settings for segmentation and computes the MPT
	(Maximum Phonation Time) from the selected interval.

	Attributes:
	    atenu_id (int): Identifier for this specific a tenu task instance.
	    is_signal_selection_movable (bool): Whether the selection can be moved.
	    is_signal_selection_manually_resizable (bool): Whether selection can be resized manually.
	    signal_selection_min_length (float): Minimum allowable selection duration (in seconds).
	    signal_selection_max_length (float): Maximum allowable selection duration (in seconds).
	    mpt (Output): Output container for the Maximum Phonation Time value.
	"""

	atenu_id: int
	is_signal_selection_movable: bool
	is_signal_selection_manually_resizable: bool
	signal_selection_min_length: float
	signal_selection_max_length: float
	mpt: Output

	def __init__(
		self,
		atenu_id: int,
		speaker_code: str,
		session_date: str,
	):
		super().__init__(
			speaker_code,
			f"PneumoPhonatoire_aTenu{atenu_id}",
			session_date,
		)
		self.mpt = Output("MPT", "")
		self.atenu_id = atenu_id
		self.is_signal_selection_movable = True
		self.is_signal_selection_manually_resizable = True
		self.signal_selection_min_length = 0.1
		self.signal_selection_max_length = 100

	@override
	def get_results(self) -> dict[str, Union[float, str]]:
		"""
		Returns the results as a dictionary with the current interval duration
		associated with the MPT candidate identifier.
		"""
		return {
			f"mpt_candidate_{self.atenu_id}": self.get_interval_duration(),
		}

	@override
	def get_io(self) -> Info:
		"""
		Creates and returns an Info object containing the input and output info boxes.

		- The first InfoBox provides instructions for precise segmentation adjustment
		at the beginning and end of the sustained /a/ sound.
		- The second InfoBox displays results dynamically, including the MPT output.

		Returns:
			Info: An Info object containing the configured InfoBoxes.
		"""
		info_box = InfoBox(
			f"Segmentation aTenu-{self.atenu_id}",
			"Ajuster les frontières, précisément, au début et à :a fin du /a/ tenu."
			"Ne pas inclure ce qui suit une éventuelle pause respiratoire.",
		)

		resultats = InfoBox("Résultats", dynamic_content=[self.mpt])

		info = Info()

		info.add_infobox(info_box)
		info.add_infobox(resultats)

		return info

	@override
	def update_io(self):
		"""
		Updates the MPT output value based on the current interval duration.
		"""
		self.mpt.update_value(self.get_interval_duration())
