################################################################################################################
## This script opens all sound files in a directory, one by one, checks if a TextGrid with the same name 
## already exists. If it doesn't, it creates a TextGrid with specified tier names, automatically inserts the
## correct number of intervals and their labels,  opens sound file and TextGrid together into the editor where 
## boundaries of the interval have to be adjusted to audio signal.
## Once it goes through all the files in the directory, it measures duration and different F0 measures of the 
## annotated intervals on each of the tiers in all TextGrind files and writes the measures into a result file.
##
## The script uses parts of code written by Celine De Looze (script Get_Phrases_register.praat) and N. Audibert.
## The script uses procedure split written by Atria (2014).
## The script was written for analyzing data for project MonPaGe. May 2016.
## Revised in September 2016
################################################################################################################

# Create a dialog box and enter the directory location with the sound files, the type of sound files and
# the name of the textgrid tier(s).

form Enter directory
# Enter full path to the directory where the recordings are stored
# IMPORTANT: Insert a SLASH AT THE END of the directory name!!!(backslash \ for Windows, forward slash / for Mac)
# IMPORTANT: Textgrid tier names have to be enclosed in " ".
	sentence IMPORTANT_READ Insert \ (for Windows) or / (for Mac) at the end of directory name.
	sentence Directory 
	sentence Filetype wav
	sentence TierName "Phrase Comments"
	sentence Resultfile Results_prosody_Phrase
endform

#####################################################################################################################
## This part of the script creates textgrids, inserts all the necessary boundaries and labels the relevant intervals.
# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings

for i from 1 to numberOfFiles

	# Select the first filename from the list.  
	select Strings fileList
	soundFile$ = Get string... i
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")
	sentenceType$ = right$ (fileName$, 12)
	
	# Get total duration of the sound file
	totalDuration = Get total duration

	# Check if TextGrid exists for the file
	textgrid_name$ = directory$ + fileName$ + ".TextGrid"
	if fileReadable (textgrid_name$)
	else

		# Create a TextGrid for the selected sound file.
		To TextGrid... 'TierName$'

		#Insert boundaries at a setspecified and label the created interval.
		if sentenceType$ = "aurie-Neutre"
			Insert boundary: 1, 0.200
			Set interval text: 1, 2, "Laurie Neutre"
			Insert boundary: 1, totalDuration - 0.200
		elsif sentenceType$ = "rie-Question"
			Insert boundary: 1, 0.200
			Set interval text: 1, 2, "Laurie Question"
			Insert boundary: 1, totalDuration - 0.200
		elsif sentenceType$ = "lanie-Neutre"
			Insert boundary: 1, 0.200
			Set interval text: 1, 2, "Melanie Neutre"
			Insert boundary: 1, totalDuration - 0.200
		elsif sentenceType$ = "nie-Question"
			Insert boundary: 1, 0.200
			Set interval text: 1, 2, "Melanie Question"
			Insert boundary: 1, totalDuration - 0.200
		endif

		# TextGrid is saved with the same filename and in the same directory as the sound files.
		minus Sound 'fileName$'
		Write to text file... 'directory$''fileName$'.TextGrid

	# End the loop for the selected file and go on to the next file.
	endif
	select all
	minus Strings fileList
	Remove
endfor

############################################################################
## This part of the script gives user a hand to make changes to annotations.
# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings

for i from 1 to numberOfFiles

	# Select the first filename from the list.  
	select Strings fileList
	soundFile$ = Get string... i
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")

	# Open a matching textgrid file
	Read from file... 'directory$''fileName$'.TextGrid
	select TextGrid 'fileName$'

	# Select sound object together with textgrid and open.
	# The script will pause once a sound and tegrid files are opened. Insert boundaries and annotations.
	# Click CONTINUE to move to the next sound-textgrid pair.
	plus Sound 'fileName$'
	Edit
	pause  Insert boundaries and annotations. 

	# TextGrid is saved with the same filename and in the same directory as the sound files.
	minus Sound 'fileName$'
	Write to text file... 'directory$''fileName$'.TextGrid

	select all
	minus Strings fileList
	Remove
endfor

#################################################################################
## This part of the script performs sound file analysis and extracts the measures.
# Create results file and the header for the columns
fileappend "'directory$''resultfile$'.txt" Speaker'tab$'Sex'tab$'Language'tab$'FileName'tab$'Module'tab$'Task'tab$'TierName'tab$'Label'tab$'Comment'tab$'Start'tab$'End'tab$'Duration'tab$'MedianF0_2nd'tab$'MedianF0_3rd'tab$'MedianF0diff'tab$'sdF0_2nd'tab$'sdF0_3rd'tab$'sdF0diff'tab$'coefVarF0_2nd'tab$'coefVarF0_3rd'tab$'coefVarF0diff'newline$'

# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings
for n from 1 to numberOfFiles

	# Start by selecting the first file from the list and go through the list.  
	select Strings fileList
	soundFile$ = Get string: n
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")

	#Sound to pitch
	select Sound 'fileName$'
	pitch_step = 0.01
	To Pitch... 'pitch_step' 60 600
	myPitch=selected("Pitch")
	myPitch$=selected$("Pitch")
	minimum_f0= Get minimum... 0 0 Hertz Parabolic
	maximum_f0= Get maximum... 0 0 Hertz Parabolic
	q65 = Get quantile... 0.0 0.0 0.65 Hertz
	q15 = Get quantile... 0.0 0.0 0.15 Hertz
	
	max_f0 = 10*ceiling((1.92*q65)/10)
	min_f0 = 10*floor((0.83*q15)/10)
	
	select Sound 'fileName$'
	To Pitch... 'pitch_step' 'min_f0' 'max_f0'

	# Open a matching textgrid file
	Read from file... 'directory$''fileName$'.TextGrid
	select TextGrid 'fileName$'

	# Call a procedure @split from a praat script positioned at the end of this file.
	# Extract speaker code, sex, language, module and task from the file name.

	@split ("_", fileName$)
	for i to split.length
		str$[i] = split.array$[i]
	endfor


######################### Parametrage en fonction du nom du fichier ###########
	speaker$ = ""
	sex$ = ""
	language$ = ""
	module$ = "module"
	task$ = "stimuli"

	if split.length == 8
		speaker$ = split.array$[3]
		sex$ = split.array$[2]
		language$ = split.array$[1]
		module$ = split.array$[7]
		task$ = split.array$[8]

	elsif split.length == 6
		speaker$ = split.array$[1]
		module$ = split.array$[5]
		task$ = split.array$[6]

	endif


########################################################################

	# Get measures of each interval on each tier
	numberOfTiers = Get number of tiers

	for t to numberOfTiers-1
	 	tierName$ = Get tier name: t
		numberOfIntervals = Get number of intervals: t

		for j to numberOfIntervals
		intervalLabel$ = Get label of interval: t, j

			if intervalLabel$ <> ""
				intervalStart = Get starting point: t, j
				intervalEnd = Get end point: t, j
				intervalDuration = intervalEnd - intervalStart
				interval2nd = intervalStart + (intervalDuration / 3)
				interval3rd = intervalStart + 2*(intervalDuration / 3)

				#Check if there is a comment in a Comments tier for any of the annotated intervals on other tiers.
				numberOfIntervalsComments = Get number of intervals: numberOfTiers
				
				comment$ = ""

				for c to numberOfIntervalsComments
					intervalCommentLabel$ = Get label of interval: numberOfTiers, c
				
					if intervalCommentLabel$ <> ""
						intervalStartComments = Get starting point: numberOfTiers, c
						intervalEndComments = Get end point: numberOfTiers, c

						if not (  (intervalEndComments < intervalStart) or (intervalStartComments > intervalEnd)  )
							comment$ = comment$ + " " + intervalCommentLabel$
						endif
					endif
				endfor

				select Pitch 'fileName$'
				meanF02nd = Get mean... interval2nd interval3rd Hertz
				meanF03rd = Get mean... interval3rd intervalEnd Hertz
				medianF02nd = Get quantile... interval2nd interval3rd 0.5 Hertz
				medianF03rd = Get quantile... interval3rd intervalEnd 0.5 Hertz
				medianF0dif = medianF03rd - medianF02nd
				sdF02nd = Get standard deviation... interval2nd interval3rd Hertz
				sdF03rd = Get standard deviation... interval3rd intervalEnd Hertz
				sdF0dif = sdF03rd - sdF02nd
				coefVarF02nd = sdF02nd / meanF02nd
				coefVarF03rd = sdF03rd / meanF03rd
				coefVarF0dif = coefVarF03rd - coefVarF02nd

				fileappend "'directory$''resultfile$'.txt" 'speaker$''tab$''sex$''tab$''language$''tab$''fileName$''tab$''module$''tab$''task$''tab$''tierName$''tab$''intervalLabel$''tab$''comment$''tab$''intervalStart:3''tab$''intervalEnd:3''tab$''intervalDuration:3''tab$''medianF02nd:3''tab$''medianF03rd:3''tab$''medianF0dif:3''tab$''sdF02nd:3''tab$''sdF03rd:3''tab$''sdF0dif:3''tab$''coefVarF02nd:3''tab$''coefVarF03rd:3''tab$''coefVarF0dif:3''newline$'
				select TextGrid 'fileName$'
			endif
		endfor
	endfor
endfor

# Clean up the Praat objects window.
select all
Remove

##################################################################
# Split .str$ on .sep$ and store each found element in .array$ and
# length of .array$ in .length
#
# Usage:
#  include /path/to/this/script.praat
#  [code]
#  @split (SEPARATOR, STRING)
#  for i to split.length
#    str$[i] = split.array$[i]
#  endfor
#
# where SEPARATOR is a separator string and STRING is a string to
# separate.
#
# If string$ = "hello world", then after
# @split (" ", string$)
# split.array$[1] contains "hello" and split.array$[2] contains "world"
#
# Notes:
# - Since .length stores the number of items separated by a string, it is always
#   larger than the amount of occurences of that string by one, which means
#   it can be used to count occurences as well.
# - This script has been changed to use the new Praat syntax, which
#   started being introduced after v.5.3.44. It can be made to work with
#   the old syntax by replacing the definition to
#
#       procedure split .sep$ .str$
#
#   and, with the same example as above, calling it with
#
#       call split " " 'string$'
#
# Written by Jose J. Atria (28 February 2012)
# Last updated: 20 February 2014
# This script is free software: you can redistribute it and/or modify 
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# A copy of the GNU General Public License is available at
# <http://www.gnu.org/licenses/>.

procedure split (.sep$, .str$)
  .seplen = length(.sep$) 
  .length = 0
  repeat
    .strlen = length(.str$)
    .sep = index(.str$, .sep$)
    if .sep > 0
      .part$ = left$(.str$, .sep-1)
      .str$ = mid$(.str$, .sep+.seplen, .strlen)
    else
      .part$ = .str$
    endif
    .length = .length+1
    .array$[.length] = .part$
  until .sep = 0
endproc