################################################################################################################
## This script opens all sound files in a directory, one by one, checks if a TextGrid with the same name 
## already exists. If it doesn't, it creates a TextGrid with specified tier names, automatically inserts the
## correct number of intervals and their labels,  opens sound file and TextGrid together into the editor where 
## boundaries of the interval have to be adjusted to audio signal.
## Once it goes through all the files in the directory, it measures duration, F0, formants and intensity of the 
## annotated intervals on each of the tiers in all TextGrind files and writes the measures into a result file.
##
## The script uses parts of code written by Celine De Looze (script Get_Phrases_register.praat) and N. Audibert.
## The script uses procedure split written by Atria (2014) and procedure filtrage written by Gendrot & Adda Decker (2005). 
## The script uses parts of the code from the "markbysyllable" script from http://www.praatvocaltoolkit.com/index.html
## The script was written for analyzing data for project MonPaGe. May 2016.
## Revised in November 2016
################################################################################################################

# Create a dialog box and enter the directory location with the sound files, the type of sound files and
# the name of the textgrid tier(s).

form Enter directory
# Enter full path to the directory where the recordings are stored
# IMPORTANT: Insert a SLASH AT THE END of the directory name!!!(backslash \ for Windows, forward slash / for Mac)
# IMPORTANT: Textgrid tier names have to be enclosed in " ".
	sentence IMPORTANT_READ Insert \ (for Windows) or / (for Mac) at the end of directory name.
	sentence Directory 
	sentence Filetype wav
	sentence TierName "Vowel Comments"
	sentence Resultfile Results_vowels_filtrage
endform

#####################################################################################################################
## This part of the script creates textgrids, inserts all the necessary boundaries and labels the relevant intervals.
# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings

for b from 1 to numberOfFiles

	# Select the first filename from the list.  
	select Strings fileList
	soundFile$ = Get string... b
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")
	vowel$ = right$ (fileName$, 2)
	
	# Get total duration of the sound file
	totalDuration = Get total duration

	# Check if TextGrid exists for the file
	textgrid_name$ = directory$ + fileName$ + ".TextGrid"
	select Sound 'fileName$'

	if fileReadable (textgrid_name$)
	else

		# Create a TextGrid for the selected sound file.
		To TextGrid... 'TierName$'
		
############################################
		# Set intervals	using intensity
			select Sound 'fileName$'
			fileName = selected ("Sound")
			Subtract mean
			tmp = Filter (pass Hann band)... 60 0 20
			clip = Get absolute extremum... 0 0 None
				if clip = undefined
					clip = 0
				endif
					clip = 'clip:4'
				if clip >= 1
					Scale... 0.9999
				endif
				select fileName
				Formula... Object_'tmp'[]
				select tmp
				Remove
				select fileName
				stt = Get start time
					if stt <> 0
						dur = Get total duration
						Scale times to... 0 dur
					endif
				Scale... 0.9999
				dur = Get total duration
				intensity = To Intensity... 40 0.05 0
				intensitytier = Down to IntensityTier
				tableofreal = Down to TableOfReal
				pnts = Get number of rows
				for i from 1 to pnts	
					w = i + 1
					z = i - 1
					pnt'i' = Get value... 'i' 1
					if z <>0
						pntprev'i' = Get value... 'z' 1
					endif
					if i <>pnts 
						pntnext'i' = Get value... 'w' 1
					endif
					db'i' = Get value... 'i' 2
				endfor
				thld = 72
				timer = 0
				select TextGrid 'fileName$'
				for i from 1 to pnts
					tPnt = pnt'i'
					iPrev = i - 1
					iNext = i + 1
					db = db'i'

					if i<>1
						dbPrev = db'iPrev'
					else
						dbPrev = db
					endif
					if i<>pnts
						dbNext = db'iNext'
					else
						dbNext = db
					endif
					if db>thld and dbPrev<thld 
						if timer < 6
							Insert boundary... 1 pntprev'i'
							timer = timer + 1
						endif
					elsif db>thld and dbNext<thld
						if timer < 6
							Insert boundary... 1 pntnext'i'
							timer = timer + 1
						endif
					endif

				endfor
		#Set 6 boundaries if the previous procedures failed to do it
		if timer = 0
			Insert boundary... 1 totalDuration/7
			Insert boundary... 1 2*(totalDuration/7)
			Insert boundary... 1 3*(totalDuration/7)
			Insert boundary... 1 4*(totalDuration/7)			Insert boundary... 1 5*(totalDuration/7)
			Insert boundary... 1 6*(totalDuration/7)
		else
			failures = 6-timer
			if failures <> 0
				for f from 1 to failures
					intervalFailure = Get number of intervals... 1
					timeFailure = Get starting point... 1 intervalFailure
					Insert boundary... 1 timeFailure+((totalDuration-timeFailure)/2)
				endfor
			endif 
		endif
		#Provides annotation for intervals of interest
		if vowel$ = "pa"
			Set interval text: 1, 2, "a1"
			Set interval text: 1, 4, "a2"
			Set interval text: 1, 6, "a3"
		elsif vowel$ = "pe"
			Set interval text: 1, 2, "e1"
			Set interval text: 1, 4, "e2"
			Set interval text: 1, 6, "e3"
		elsif vowel$ = "Ep"
			Set interval text: 1, 2, "E1"
			Set interval text: 1, 4, "E2"
			Set interval text: 1, 6, "E3"
		elsif vowel$ = "pi"
			Set interval text: 1, 2, "i1"
			Set interval text: 1, 4, "i2"
			Set interval text: 1, 6, "i3"
		elsif vowel$ = "au"			
			Set interval text: 1, 2, "au1"
			Set interval text: 1, 4, "au2"
			Set interval text: 1, 6, "au3"
		elsif vowel$ = "op"
			Set interval text: 1, 2, "o1"
			Set interval text: 1, 4, "o2"
			Set interval text: 1, 6, "o3"
		elsif vowel$ = "ou"
			Set interval text: 1, 2, "ou1"
			Set interval text: 1, 4, "ou2"
			Set interval text: 1, 6, "ou3"
		endif

		# TextGrid is saved with the same filename and in the same directory as the sound files.
		minus Sound 'fileName$'
		Write to text file... 'directory$''fileName$'.TextGrid

	# End the loop for the selected file and go on to the next file.
	endif
	select all
	minus Strings fileList
	Remove
endfor

#############################################################################
## This part of the script gives user a hand to make changes to annotations.
# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings

for i from 1 to numberOfFiles

	# Select the first filename from the list.  
	select Strings fileList
	soundFile$ = Get string... i
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")

	# Open a matching textgrid file
	Read from file... 'directory$''fileName$'.TextGrid
	select TextGrid 'fileName$'

	# Select sound object together with textgrid and open.
	# The script will pause once a sound and tegrid files are opened. Insert boundaries and annotations.
	# Click CONTINUE to move to the next sound-textgrid pair.
	plus Sound 'fileName$'
	Edit
	pause  Insert boundaries and annotations. 

	# TextGrid is saved with the same filename and in the same directory as the sound files.
	minus Sound 'fileName$'
	Write to text file... 'directory$''fileName$'.TextGrid

	select all
	minus Strings fileList
	Remove
endfor

##################################################################################
## This part of the script performs sound file analysis and extracts the measures.
# Create results file and the header for the columns
fileappend "'directory$''resultfile$'.txt" Speaker'tab$'Sex'tab$'Language'tab$'FileName'tab$'Module'tab$'Task'tab$'TierName'tab$'Label'tab$'Comment'tab$'Start'tab$'End'tab$'Duration'tab$'MeanF0'tab$'MedianF0'tab$'sdF0'tab$'coefVarF0'tab$'FormantMeasures'tab$'MeanF1'tab$'MedianF1'tab$'sdF1'tab$'coefVarF1'tab$'MeanF2'tab$'MedianF2'tab$'sdF2'tab$'coefVarF2'tab$'MeanF3'tab$'MedianF3'tab$'sdF3'tab$'coefVarF3'tab$'F0_Start'tab$'F0_10%'tab$'F0_20%'tab$'F0_30%'tab$'F0_40%'tab$'F0_50%'tab$'F0_60%'tab$'F0_70%'tab$'F0_80%'tab$'F0_90%'tab$'F0_End'tab$'F1_Start'tab$'F1_10%'tab$'F1_20%'tab$'F1_30%'tab$'F1_40%'tab$'F1_50%'tab$'F1_60%'tab$'F1_70%'tab$'F1_80%'tab$'F1_90%'tab$'F1_End'tab$'F2_Start'tab$'F2_10%'tab$'F2_20%'tab$'F2_30%'tab$'F2_40%'tab$'F2_50%'tab$'F2_60%'tab$'F2_70%'tab$'F2_80%'tab$'F2_90%'tab$'F2_End'tab$'F3_Start'tab$'F3_10%'tab$'F3_20%'tab$'F3_30%'tab$'F3_40%'tab$'F3_50%'tab$'F3_60%'tab$'F3_70%'tab$'F3_80%'tab$'F3_90%'tab$'F3_End'tab$'MeanIntensity'tab$'MedianIntensity'tab$'sdIntensity'tab$'coefVarIntensity'tab$'MeanHNR'tab$'sdHNR'tab$'coefVarHNR'newline$'

# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings
for n from 1 to numberOfFiles

	# Start by selecting the first file from the list and go through the list.  
	select Strings fileList
	soundFile$ = Get string: n
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")

	#Sound to pitch
	select Sound 'fileName$'
	pitch_step = 0.01
	To Pitch... 'pitch_step' 60 600
	myPitch=selected("Pitch")
	myPitch$=selected$("Pitch")
	minimum_f0= Get minimum... 0 0 Hertz Parabolic
	maximum_f0= Get maximum... 0 0 Hertz Parabolic
	q65 = Get quantile... 0.0 0.0 0.65 Hertz
	q15 = Get quantile... 0.0 0.0 0.15 Hertz
	
	max_f0 = 10*ceiling((1.92*q65)/10)
	min_f0 = 10*floor((0.83*q15)/10)
	
	select Sound 'fileName$'
	To Pitch... 'pitch_step' 'min_f0' 'max_f0'

	select Sound 'fileName$'
	intensity = do ("To Intensity...", min_f0, 0, "yes")

	# Open a matching textgrid file
	Read from file... 'directory$''fileName$'.TextGrid
	select TextGrid 'fileName$'

	# Call a procedure @split from a praat script positioned at the end of this file.
	# Extract speaker code, sex, language, module and task from the file name.

	@split ("_", fileName$)
	for i to split.length
		str$[i] = split.array$[i]
	endfor


######################### Parametrage en fonction du nom du fichier ###########
	speaker$ = ""
	sex$ = ""
	language$ = ""
	module$ = "module"
	task$ = "stimuli"

	if split.length == 8
		speaker$ = split.array$[3]
		sex$ = split.array$[2]
		language$ = split.array$[1]
		module$ = split.array$[7]
		task$ = split.array$[8]

	elsif split.length == 6
		speaker$ = split.array$[1]
		module$ = split.array$[5]
		task$ = split.array$[6]

	endif

	while (sex$ <> "H") and (sex$ <> "F")
		beginPause: "Informations necessaires introuvables dans le nom du fichier"
			comment: fileName$
			comment: "Sexe (H/F):" 
			text: "sex", sex$
		clicked = endPause: "Ok", 1
	endwhile


########################################################################

	# Get measures of each interval on each tier
	numberOfTiers = Get number of tiers

	for t to numberOfTiers-1
	 	tierName$ = Get tier name: t
		numberOfIntervals = Get number of intervals: t

		for j to numberOfIntervals
		intervalLabel$ = Get label of interval: t, j

			if intervalLabel$ <> ""
				intervalStart = Get starting point: t, j
				intervalEnd = Get end point: t, j
				intervalDuration = intervalEnd - intervalStart
				interval2Q = intervalStart + (intervalDuration / 4)
				interval4Q = intervalEnd - (intervalDuration / 4)

				# Define time points at every 10% of interval durations to extract the measures.
				intervalDuration10 = (intervalDuration * 10) / 100
				timeStart = intervalStart + 0.005
				time10 = intervalStart + intervalDuration10
				time20 = intervalStart + (2 * intervalDuration10)
				time30 = intervalStart + (3 * intervalDuration10)
				time40 = intervalStart + (4 * intervalDuration10)
				time50 = intervalStart + (5 * intervalDuration10)
				time60 = intervalStart + (6 * intervalDuration10)
				time70 = intervalStart + (7 * intervalDuration10)
				time80 = intervalStart + (8 * intervalDuration10)
				time90 = intervalStart + (9 * intervalDuration10)
				timeEnd = intervalEnd - 0.005

				# Extract F0
				select Pitch 'fileName$'
				meanF0 = Get mean... intervalStart intervalEnd Hertz
				medianF0 = Get quantile... intervalStart intervalEnd 0.5 Hertz
				sdF0 = Get standard deviation... intervalStart intervalEnd Hertz
				coefVarF0 = sdF0 / meanF0
				minF0 = Get minimum... intervalStart intervalEnd Hertz Parabolic
				f0_Start = Get value at time... timeStart Hertz Linear
				f0_10 = Get value at time... time10 Hertz Linear
				f0_20 = Get value at time... time20 Hertz Linear
				f0_30 = Get value at time... time30 Hertz Linear
				f0_40 = Get value at time... time40 Hertz Linear
				f0_50 = Get value at time... time50 Hertz Linear
				f0_60 = Get value at time... time60 Hertz Linear
				f0_70 = Get value at time... time70 Hertz Linear
				f0_80 = Get value at time... time80 Hertz Linear
				f0_90 = Get value at time... time90 Hertz Linear
				f0_End = Get value at time... timeEnd Hertz Linear

				# Extract formants
				select Sound 'fileName$'
				if sex$ = "H"
					To Formant (burg)... 0 5 5000 0.025 50
					select Formant 'fileName$'
					meanF1 = Get mean... 1 interval2Q interval4Q Hertz
					medianF1 = Get quantile... 1 interval2Q interval4Q Hertz 0.5
					sdF1 = Get standard deviation... 1 interval2Q interval4Q Hertz
					coefVarF1 = sdF1 / meanF1
					meanF2 = Get mean... 2 interval2Q interval4Q Hertz
					medianF2 = Get quantile... 2 interval2Q interval4Q Hertz 0.5
					sdF2 = Get standard deviation... 2 interval2Q interval4Q Hertz
					coefVarF2 = sdF2 / meanF2
					meanF3 = Get mean... 3 interval2Q interval4Q Hertz
					medianF3 = Get quantile... 3 interval2Q interval4Q Hertz 0.5
					sdF3 = Get standard deviation... 3 interval2Q interval4Q Hertz
					coefVarF3 = sdF3 / meanF3

					@filtrage

					f1_Start = Get value at time... 1 timeStart Hertz Linear
					f1_10 = Get value at time... 1 time10 Hertz Linear
					f1_20 = Get value at time... 1 time20 Hertz Linear
					f1_30 = Get value at time... 1 time30 Hertz Linear
					f1_40 = Get value at time... 1 time40 Hertz Linear
					f1_50 = Get value at time... 1 time50 Hertz Linear
					f1_60 = Get value at time... 1 time60 Hertz Linear
					f1_70 = Get value at time... 1 time70 Hertz Linear
					f1_80 = Get value at time... 1 time80 Hertz Linear
					f1_90 = Get value at time... 1 time90 Hertz Linear
					f1_End = Get value at time... 1 timeEnd Hertz Linear


					f2_Start = Get value at time... 2 timeStart Hertz Linear
					f2_10 = Get value at time... 2 time10 Hertz Linear
					f2_20 = Get value at time... 2 time20 Hertz Linear
					f2_30 = Get value at time... 2 time30 Hertz Linear
					f2_40 = Get value at time... 2 time40 Hertz Linear
					f2_50 = Get value at time... 2 time50 Hertz Linear
					f2_60 = Get value at time... 2 time60 Hertz Linear
					f2_70 = Get value at time... 2 time70 Hertz Linear
					f2_80 = Get value at time... 2 time80 Hertz Linear
					f2_90 = Get value at time... 2 time90 Hertz Linear
					f2_End = Get value at time... 2 timeEnd Hertz Linear

					f3_Start = Get value at time... 3 timeStart Hertz Linear
					f3_10 = Get value at time... 3 time10 Hertz Linear
					f3_20 = Get value at time... 3 time20 Hertz Linear
					f3_30 = Get value at time... 3 time30 Hertz Linear
					f3_40 = Get value at time... 3 time40 Hertz Linear
					f3_50 = Get value at time... 3 time50 Hertz Linear
					f3_60 = Get value at time... 3 time60 Hertz Linear
					f3_70 = Get value at time... 3 time70 Hertz Linear
					f3_80 = Get value at time... 3 time80 Hertz Linear
					f3_90 = Get value at time... 3 time90 Hertz Linear
					f3_End = Get value at time... 3 timeEnd Hertz Linear

				elsif sex$ = "F"
					To Formant (burg)... 0 5 5500 0.025 50
					select Formant 'fileName$'
					meanF1 = Get mean... 1 interval2Q interval4Q Hertz
					medianF1 = Get quantile... 1 interval2Q interval4Q Hertz 0.5
					sdF1 = Get standard deviation... 1 interval2Q interval4Q Hertz
					coefVarF1 = sdF1 / meanF1
					meanF2 = Get mean... 2 interval2Q interval4Q Hertz
					medianF2 = Get quantile... 2 interval2Q interval4Q Hertz 0.5
					sdF2 = Get standard deviation... 2 interval2Q interval4Q Hertz
					coefVarF2 = sdF2 / meanF2
					meanF3 = Get mean... 3 interval2Q interval4Q Hertz
					medianF3 = Get quantile... 3 interval2Q interval4Q Hertz 0.5
					sdF3 = Get standard deviation... 3 interval2Q interval4Q Hertz
					coefVarF3 = sdF3 / meanF3

					@filtrage

					f1_Start = Get value at time... 1 timeStart Hertz Linear
					f1_10 = Get value at time... 1 time10 Hertz Linear
					f1_20 = Get value at time... 1 time20 Hertz Linear
					f1_30 = Get value at time... 1 time30 Hertz Linear
					f1_40 = Get value at time... 1 time40 Hertz Linear
					f1_50 = Get value at time... 1 time50 Hertz Linear
					f1_60 = Get value at time... 1 time60 Hertz Linear
					f1_70 = Get value at time... 1 time70 Hertz Linear
					f1_80 = Get value at time... 1 time80 Hertz Linear
					f1_90 = Get value at time... 1 time90 Hertz Linear
					f1_End = Get value at time... 1 timeEnd Hertz Linear

					f2_Start = Get value at time... 2 timeStart Hertz Linear
					f2_10 = Get value at time... 2 time10 Hertz Linear
					f2_20 = Get value at time... 2 time20 Hertz Linear
					f2_30 = Get value at time... 2 time30 Hertz Linear
					f2_40 = Get value at time... 2 time40 Hertz Linear
					f2_50 = Get value at time... 2 time50 Hertz Linear
					f2_60 = Get value at time... 2 time60 Hertz Linear
					f2_70 = Get value at time... 2 time70 Hertz Linear
					f2_80 = Get value at time... 2 time80 Hertz Linear
					f2_90 = Get value at time... 2 time90 Hertz Linear
					f2_End = Get value at time... 2 timeEnd Hertz Linear

					f3_Start = Get value at time... 3 timeStart Hertz Linear
					f3_10 = Get value at time... 3 time10 Hertz Linear
					f3_20 = Get value at time... 3 time20 Hertz Linear
					f3_30 = Get value at time... 3 time30 Hertz Linear
					f3_40 = Get value at time... 3 time40 Hertz Linear
					f3_50 = Get value at time... 3 time50 Hertz Linear
					f3_60 = Get value at time... 3 time60 Hertz Linear
					f3_70 = Get value at time... 3 time70 Hertz Linear
					f3_80 = Get value at time... 3 time80 Hertz Linear
					f3_90 = Get value at time... 3 time90 Hertz Linear
					f3_End = Get value at time... 3 timeEnd Hertz Linear
				endif

				# Extract intensity
				select Intensity 'fileName$'
				medianIntensity = Get quantile... intervalStart intervalEnd 0.5
				meanIntensity = Get mean... intervalStart intervalEnd
				sdIntensity = Get standard deviation... intervalStart intervalEnd
				coefVarIntensity = sdIntensity / meanIntensity

				# Extract HNR
				select Sound 'fileName$'
				if minF0 = undefined
					To Harmonicity (cc)... 0.01 75 0.1 1
				else
					To Harmonicity (cc)... 0.01 minF0 0.1 1
				endif
				meanHNR = Get mean... 0 0
				sdHNR = Get standard deviation... 0 0
				coefVarHNR = sdHNR / meanHNR

				#Check if there is a comment in a Comments tier for any of the annotated intervals on other tiers.
				select TextGrid 'fileName$'
				numberOfIntervalsComments = Get number of intervals: numberOfTiers
				
				comment$ = ""

				for c to numberOfIntervalsComments
					intervalCommentLabel$ = Get label of interval: numberOfTiers, c
				
					if intervalCommentLabel$ <> ""
						intervalStartComments = Get starting point: numberOfTiers, c
						intervalEndComments = Get end point: numberOfTiers, c

						if not (  (intervalEndComments < intervalStart) or (intervalStartComments > intervalEnd)  )
							comment$ = comment$ + " " + intervalCommentLabel$
						endif
					endif
				endfor


				fileappend "'directory$''resultfile$'.txt" 'speaker$''tab$''sex$''tab$''language$''tab$''fileName$''tab$''module$''tab$''task$''tab$''tierName$''tab$''intervalLabel$''tab$''comment$''tab$''intervalStart:3''tab$''intervalEnd:3''tab$''intervalDuration:3''tab$''meanF0:3''tab$''medianF0:3''tab$''sdF0:3''tab$''coefVarF0:3''tab$''mesures_formants$''tab$''meanF1:3''tab$''medianF1:3''tab$''sdF1:3''tab$''coefVarF1:3''tab$''meanF2:3''tab$''medianF2:3''tab$''sdF2:3''tab$''coefVarF2:3''tab$''meanF3:3''tab$''medianF3:3''tab$''sdF3:3''tab$''coefVarF3:3''tab$''f0_Start:3''tab$''f0_10:3''tab$''f0_20:3''tab$''f0_30:3''tab$''f0_40:3''tab$''f0_50:3''tab$''f0_60:3''tab$''f0_70:3''tab$''f0_80:3''tab$''f0_90:3''tab$''f0_End:3''tab$''f1_Start:3''tab$''f1_10:3''tab$''f1_20:3''tab$''f1_30:3''tab$''f1_40:3''tab$''f1_50:3''tab$''f1_60:3''tab$''f1_70:3''tab$''f1_80:3''tab$''f1_90:3''tab$''f1_End:3''tab$''f2_Start:3''tab$''f2_10:3''tab$''f2_20:3''tab$''f2_30:3''tab$''f2_40:3''tab$''f2_50:3''tab$''f2_60:3''tab$''f2_70:3''tab$''f2_80:3''tab$''f2_90:3''tab$''f2_End:3''tab$''f3_Start:3''tab$''f3_10:3''tab$''f3_20:3''tab$''f3_30:3''tab$''f3_40:3''tab$''f3_50:3''tab$''f3_60:3''tab$''f3_70:3''tab$''f3_80:3''tab$''f3_90:3''tab$''f3_End:3''tab$''meanIntensity:3''tab$''medianIntensity:3''tab$''sdIntensity:3''tab$''coefVarIntensity:3''tab$''meanHNR:3''tab$''sdHNR:3''tab$''coefVarHNR:3''newline$'
				select TextGrid 'fileName$'
			endif
		endfor
	endfor
endfor

# Clean up the Praat objects window.
select all
Remove

##################################################################
# Split .str$ on .sep$ and store each found element in .array$ and
# length of .array$ in .length
#
# Usage:
#  include /path/to/this/script.praat
#  [code]
#  @split (SEPARATOR, STRING)
#  for i to split.length
#    str$[i] = split.array$[i]
#  endfor
#
# where SEPARATOR is a separator string and STRING is a string to
# separate.
#
# If string$ = "hello world", then after
# @split (" ", string$)
# split.array$[1] contains "hello" and split.array$[2] contains "world"
#
# Notes:
# - Since .length stores the number of items separated by a string, it is always
#   larger than the amount of occurences of that string by one, which means
#   it can be used to count occurences as well.
# - This script has been changed to use the new Praat syntax, which
#   started being introduced after v.5.3.44. It can be made to work with
#   the old syntax by replacing the definition to
#
#       procedure split .sep$ .str$
#
#   and, with the same example as above, calling it with
#
#       call split " " 'string$'
#
# Written by Jose J. Atria (28 February 2012)
# Last updated: 20 February 2014
# This script is free software: you can redistribute it and/or modify 
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# A copy of the GNU General Public License is available at
# <http://www.gnu.org/licenses/>.

procedure split (.sep$, .str$)
  .seplen = length(.sep$) 
  .length = 0
  repeat
    .strlen = length(.str$)
    .sep = index(.str$, .sep$)
    if .sep > 0
      .part$ = left$(.str$, .sep-1)
      .str$ = mid$(.str$, .sep+.seplen, .strlen)
    else
      .part$ = .str$
    endif
    .length = .length+1
    .array$[.length] = .part$
  until .sep = 0
endproc

###################################################################################
## Gendrot & Adda Decker (2005) 
procedure filtrage

mesures_formants$ = "non_mesure"
#----------------------------------------------------------------------------------
if (sex$ = "H")
#----------------------------------------------------------------------------------
	if (intervalLabel$ = "a1") or (intervalLabel$ = "a2") or (intervalLabel$ = "a3")
		if (meanF1<1000 and meanF2>800 and meanF2<2300) and meanF3>1800
		# and meanF4>2000
		mesures_formants$ = "bon"
		else	
		mesures_formants$ = "faux"
		endif
	endif
#-----------------------------------------------------------------------------------
	if (intervalLabel$ = "i1") or (intervalLabel$ = "i2") or (intervalLabel$ = "i3")
		if (meanF1<750 and meanF2>1500 and meanF2<2500) and meanF3>2000
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#------------------------------------------------------------------------------------
	if (intervalLabel$ = "au1") or (intervalLabel$ = "au2") or (intervalLabel$ = "au3")
		if (meanF1<900 and meanF2>600 and meanF2<1600)and meanF3>1500
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#-------------------------------------------------------------------------------------
	if (intervalLabel$ = "E1") or (intervalLabel$ = "E2") or (intervalLabel$ = "E3")
		if (meanF1<1000 and meanF2>1200 and meanF2<2300) and meanF3>2000
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#-------------------------------------------------------------------------------------
	if (intervalLabel$ = "ou1") or (intervalLabel$ = "ou2") or (intervalLabel$ = "ou3")
		if (meanF1<900 and meanF2>400 and meanF2<1500) and meanF3>1400
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif 
#--------------------------------------------------------------------------------------
	if (intervalLabel$ = "o1") or (intervalLabel$ = "o2") or (intervalLabel$ = "o3")
		if (meanF1<900 and meanF2>600 and meanF2<1800) and meanF3>1500
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#--------------------------------------------------------------------------------------
	if (intervalLabel$ = "e1") or (intervalLabel$ = "e2") or (intervalLabel$ = "e3")
		if (meanF1<800 and meanF2>1100 and meanF2<2400) and meanF3>2000
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#---------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------
elsif (sex$ = "F")
#---------------------------------------------------------------------------------------
	if (intervalLabel$ = "a1") or (intervalLabel$ = "a2") or (intervalLabel$ = "a3")
		if (meanF1<1100 and meanF2>900 and meanF2<2300) and meanF3>1900
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#---------------------------------------------------------------------------------------
	if (intervalLabel$ = "i1") or (intervalLabel$ = "i2") or (intervalLabel$ = "i3")
		if (meanF1<900 and meanF2>1600 and meanF2<3100) and meanF3>2500
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#----------------------------------------------------------------------------------------
	if (intervalLabel$ = "au1") or (intervalLabel$ = "au2") or (intervalLabel$ = "au3")
		if (meanF1<1000 and meanF2>600 and meanF2<1600) and meanF3>2100
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#----------------------------------------------------------------------------------------
	if (intervalLabel$ = "E1") or (intervalLabel$ = "E2") or (intervalLabel$ = "E3")
		if (meanF1<1100 and meanF2>1400 and meanF2<2700) and meanF3>2000
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#----------------------------------------------------------------------------------------
	if (intervalLabel$ = "ou1") or (intervalLabel$ = "ou2") or (intervalLabel$ = "ou3")
		if (meanF1<1000 and meanF2>400 and meanF2<1500) and meanF3>1800
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#-----------------------------------------------------------------------------------------
	if (intervalLabel$ = "o1") or (intervalLabel$ = "o2") or (intervalLabel$ = "o3")
		if (meanF1<1000 and meanF2>600 and meanF2<2000) and meanF3>2000
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#-----------------------------------------------------------------------------------------
	if (intervalLabel$ = "e1") or (intervalLabel$ = "e2") or (intervalLabel$ = "e3")
		if (meanF1<900 and meanF2>1400 and meanF2<3000) and meanF3>2200
		# and meanF4>2000
		mesures_formants$ = "bon"
		else
		mesures_formants$ = "faux"
		endif
	endif
#-----------------------------------------------------------------------------------------
endif
#-----------------------------------------------------------------------------------------
endproc