################################################################################################################
## This script opens all sound files and matching TextGrids in a directory, pair by pair, alows making
## changes to the TextGrid, segmenting the syllables and labeling the created intervals, and saving it before 
## moving to the next pair of files.
## Once it goes through all the files in the directory, it measures duration, F0 and intensity of the annotated 
## intervals on each tier in all TextGrind files and writes the measures into a result file.
##
## The script uses parts of code written by Celine De Looze (script Get_Phrases_register.praat) and N. Audibert.
## The script was written for analyzing data for project MonPaGe. May 2016.
## Revised in September 2016
################################################################################################################

# Create a dialog box and enter the directory location with the sound files, the type of sound files and
# the name of the textgrid tier(s).

form Enter directory
# Enter full path to the directory where the recordings are stored
# IMPORTANT: Insert a SLASH AT THE END of the directory name!!!(backslash \ for Windows, forward slash / for Mac)
# IMPORTANT: Textgrid tier names have to be enclosed in " ".
	sentence IMPORTANT_READ Insert \ (for Windows) or / (for Mac) at the end of directory name.
	sentence Directory 
	sentence Filetype wav
	sentence Resultfile Results_diadoco
endform

############################################################################
## This part of the script gives user a hand to make changes to annotations.
# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings

for i from 1 to numberOfFiles

	# Select the first filename from the list.  
	select Strings fileList
	soundFile$ = Get string... i
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")

	# Open a matching textgrid file
	Read from file... 'directory$''fileName$'.TextGrid
	select TextGrid 'fileName$'

	# Select sound object together with textgrid and open.
	# The script will pause once a sound and tegrid files are opened. Insert boundaries and annotations.
	# Click CONTINUE to move to the next sound-textgrid pair.
	plus Sound 'fileName$'
	Edit
	pause  Insert boundaries and annotations. 

	# TextGrid is saved with the same filename and in the same directory as the sound files.
	minus Sound 'fileName$'
	Write to text file... 'directory$''fileName$'.TextGrid

	select all
	minus Strings fileList
	Remove

endfor

#################################################################################
## This part of the script performs sound file analysis and extracts the measures.
# Create results file and the header for the columns
fileappend "'directory$''resultfile$'.txt" Speaker'tab$'Sex'tab$'Language'tab$'FileName'tab$'Module'tab$'Task'tab$'TierName'tab$'Label'tab$'Comment'tab$'Start'tab$'End'tab$'Duration'tab$'MeanF0'tab$'MedianF0'tab$'sdF0'tab$'coefVarF0'tab$'MeanIntensity'tab$'MedianIntensity'tab$'sdIntensity'tab$'coefVarIntensity'newline$'

# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings
for n from 1 to numberOfFiles

	# Start by selecting the first file from the list and go through the list.  
	select Strings fileList
	soundFile$ = Get string: n
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")

	#Sound to pitch
	select Sound 'fileName$'
	pitch_step = 0.01
	To Pitch... 'pitch_step' 60 600
	myPitch=selected("Pitch")
	myPitch$=selected$("Pitch")
	minimum_f0= Get minimum... 0 0 Hertz Parabolic
	maximum_f0= Get maximum... 0 0 Hertz Parabolic
	q65 = Get quantile... 0.0 0.0 0.65 Hertz
	q15 = Get quantile... 0.0 0.0 0.15 Hertz
	
	max_f0 = 10*ceiling((1.92*q65)/10)
	min_f0 = 10*floor((0.83*q15)/10)
	
	select Sound 'fileName$'
	To Pitch... 'pitch_step' 'min_f0' 'max_f0'

	select Sound 'fileName$'
	intensity = do ("To Intensity...", min_f0, 0, "yes")

	# Open a matching textgrid file
	Read from file... 'directory$''fileName$'.TextGrid
	select TextGrid 'fileName$'

	module$ = "Diadoco"

	# Get measures of each interval on each tier
	numberOfTiers = Get number of tiers

	for t to numberOfTiers-1
	 	tierName$ = Get tier name: t
		numberOfIntervals = Get number of intervals: t

		for j to numberOfIntervals
		intervalLabel$ = Get label of interval: t, j

			if intervalLabel$ <> ""
				intervalStart = Get starting point: t, j
				intervalEnd = Get end point: t, j
				intervalDuration = intervalEnd - intervalStart

				#Check if there is a comment in a Comments tier for any of the annotated intervals on other tiers.
				numberOfIntervalsComments = Get number of intervals: numberOfTiers
				
				comment$ = ""

				for c to numberOfIntervalsComments
					intervalCommentLabel$ = Get label of interval: numberOfTiers, c
				
					if intervalCommentLabel$ <> ""
						intervalStartComments = Get starting point: numberOfTiers, c
						intervalEndComments = Get end point: numberOfTiers, c

						if not (  (intervalEndComments < intervalStart) or (intervalStartComments > intervalEnd)  )
							comment$ = comment$ + " " + intervalCommentLabel$
						endif
					endif
				endfor

				select Pitch 'fileName$'
				meanF0 = Get mean... intervalStart intervalEnd Hertz
				medianF0 = Get quantile... intervalStart intervalEnd 0.5 Hertz
				sdF0 = Get standard deviation... intervalStart intervalEnd Hertz
				coefVarF0 = sdF0 / meanF0
				minF0 = Get minimum... 0 0 Hertz Parabolic


				select Intensity 'fileName$'
				medianIntensity = Get quantile... intervalStart intervalEnd 0.5
				meanIntensity = Get mean... intervalStart intervalEnd
				sdIntensity = Get standard deviation... intervalStart intervalEnd
				coefVarIntensity = sdIntensity / meanIntensity


				select Sound 'fileName$'
				To Harmonicity (cc)... 0.01 minF0 0.1 1
				meanHNR = Get mean... 0 0
				sdHNR = Get standard deviation... 0 0
				coefVarHNR = sdHNR / meanHNR


				fileappend "'directory$''resultfile$'.txt" 'tab$''tab$''tab$''fileName$''tab$''module$''tab$''tab$''tierName$''tab$''intervalLabel$''tab$''comment$''tab$''intervalStart:3''tab$''intervalEnd:3''tab$''intervalDuration:3''tab$''meanF0:3''tab$''medianF0:3''tab$''sdF0:3''tab$''coefVarF0:3''tab$''meanIntensity:3''tab$''medianIntensity:3''tab$''sdIntensity:3''tab$''coefVarIntensity:3''newline$'
				select TextGrid 'fileName$'
			endif
		endfor
	endfor
endfor

# Clean up the Praat objects window.
select all
Remove
