################################################################################################################
## This script opens all sound files and matching TextGrids in a directory, pair by pair, and it produces a voice
## report of the annotated interval in all TextGrid-Sound pair files and writes the measures into a result file.
##
## The script uses parts of code written by Celine De Looze (script Get_Phrases_register.praat) and N. Audibert.
## The script uses procedure split written by Atria (2014).
## The script was written for analyzing data for project MonPaGe. May 2016.
## Revised in September 2016
################################################################################################################

# Create a dialog box and enter the directory location with the sound files, the type of sound files and
# the name of the textgrid tier(s).

form Enter directory
# Enter full path to the directory where the recordings are stored
# IMPORTANT: Insert a SLASH AT THE END of the directory name!!!(backslash \ for Windows, forward slash / for Mac)
# IMPORTANT: Textgrid tier names have to be enclosed in " ".
	sentence IMPORTANT_READ Insert \ (for Windows) or / (for Mac) at the end of directory name.
	sentence Directory 
	sentence Filetype wav
	sentence Resultfile Results_VoiceReport
endform


# Create results file and the header for the columns
fileappend "'directory$''resultfile$'.txt" Speaker'tab$'Sex'tab$'Language'tab$'FileName'tab$'Module'tab$'Task'tab$'TierName'tab$'Label'tab$'Comment'tab$'Start'tab$'End'tab$'Duration'tab$'MedianF0'tab$'MeanF0'tab$'sdF0'tab$'coefVarF0'tab$'FractionOfLocallyUnvoicedFrames'tab$'NumberOfVoiceBreaks'tab$'Jitter_local'tab$'Shimmer_local'tab$'MeanNHR'newline$'
	
# Make a list of all sound files in the directory.
Create Strings as file list... fileList 'directory$'*.'filetype$'

# Loop through all files.
numberOfFiles = Get number of strings
for n from 1 to numberOfFiles

	# Start by selecting the first file from the list and go through the list.  
	select Strings fileList
	soundFile$ = Get string: n
	Read from file... 'directory$''soundFile$'

	# A variable called "fileName$" gets the same name as the sound file.
	fileName$ = selected$ ("Sound")

	#Sound to pitch to create a Pitch object adjusted to a specific speakers
	select Sound 'fileName$'
	pitch_step = 0.01
	To Pitch... 'pitch_step' 60 600
	myPitch=selected("Pitch")
	myPitch$=selected$("Pitch")
	minimum_f0= Get minimum... 0 0 Hertz Parabolic
	maximum_f0= Get maximum... 0 0 Hertz Parabolic
	q65 = Get quantile... 0.0 0.0 0.65 Hertz
	q15 = Get quantile... 0.0 0.0 0.15 Hertz
	
	max_f0 = 10*ceiling((1.92*q65)/10)
	min_f0 = 10*floor((0.83*q15)/10)
	
	select Sound 'fileName$'
	To Pitch... 'pitch_step' 'min_f0' 'max_f0'

	select Pitch 'fileName$'
	plus Sound 'fileName$'
	pulses= To PointProcess (cc)

	
	# Open a matching textgrid file
	Read from file... 'directory$''fileName$'.TextGrid
	select TextGrid 'fileName$'

	# Call a procedure @split from a praat script positioned at the end of this file.
	# Extract speaker code, sex, language, module and task from the file name.

	@split ("_", fileName$)
	for i to split.length
		str$[i] = split.array$[i]
	endfor

######################### Parametrage en fonction du nom du fichier ###########
	speaker$ = ""
	sex$ = ""
	language$ = ""
	module$ = "module"
	task$ = "stimuli"

	if split.length == 8
		speaker$ = split.array$[3]
		sex$ = split.array$[2]
		language$ = split.array$[1]
		module$ = split.array$[7]
		task$ = split.array$[8]

	elsif split.length == 6
		speaker$ = split.array$[1]
		module$ = split.array$[5]
		task$ = split.array$[6]

	endif


########################################################################

	# Get measures of each interval on each tier
	numberOfTiers = Get number of tiers

	for t to numberOfTiers-1
	 	tierName$ = Get tier name: t
		numberOfIntervals = Get number of intervals: t

		for j to numberOfIntervals
		select TextGrid 'fileName$'
		intervalLabel$ = Get label of interval: t, j

			if intervalLabel$ <> ""
				intervalStart = Get starting point: t, j
				intervalEnd = Get end point: t, j
				intervalDuration = intervalEnd - intervalStart
				
				select Pitch 'fileName$'
				plus Sound 'fileName$'
				plus PointProcess 'fileName$'_'fileName$'

				voiceReport$= Voice report: intervalStart, intervalEnd, 75, 600, 1.3, 1.6, 0.03, 0.45
				median_pitch= extractNumber (voiceReport$, "Median pitch: ")
				mean_pitch= extractNumber (voiceReport$, "Mean pitch: ")
				stdev_pitch= extractNumber (voiceReport$, "Standard deviation: ")
				coefVar_pitch = stdev_pitch / mean_pitch
				fraction_unvoiced= extractNumber (voiceReport$, "Fraction of locally unvoiced frames: ")
				number_voice_breaks= extractNumber (voiceReport$, "Number of voice breaks: ")
				jitter_local= extractNumber (voiceReport$, "Jitter (local): ")
				shimmer_local= extractNumber (voiceReport$, "Shimmer (local): ")
				mean_hnr= extractNumber (voiceReport$, "Mean harmonics-to-noise ratio: ")

				#Check if there is a comment in a Comments tier for any of the annotated intervals on other tiers.
				select TextGrid 'fileName$'
				numberOfIntervalsComments = Get number of intervals: numberOfTiers
				comment$ = ""
				for c to numberOfIntervalsComments
					intervalCommentLabel$ = Get label of interval: numberOfTiers, c
				
					if intervalCommentLabel$ <> ""
						intervalStartComments = Get starting point: numberOfTiers, c
						intervalEndComments = Get end point: numberOfTiers, c

						if not (  (intervalEndComments < intervalStart) or (intervalStartComments > intervalEnd)  )
							comment$ = comment$ + " " + intervalCommentLabel$
						endif
					endif
				endfor

				fileappend "'directory$''resultfile$'.txt" 'speaker$''tab$''sex$''tab$''language$''tab$''fileName$''tab$''module$''tab$''task$''tab$''tierName$''tab$''intervalLabel$''tab$''comment$''tab$''intervalStart:3''tab$''intervalEnd:3''tab$''intervalDuration:3''tab$''median_pitch:3''tab$''mean_pitch:3''tab$''stdev_pitch:3''tab$''coefVar_pitch:''tab$''fraction_unvoiced''tab$''number_voice_breaks''tab$''jitter_local''tab$''shimmer_local''tab$''mean_hnr''newline$'
			
			endif
		endfor
	endfor
endfor

# Clean up the Praat objects window.
select all
Remove

# Split .str$ on .sep$ and store each found element in .array$ and
# length of .array$ in .length
#
# Usage:
#  include /path/to/this/script.praat
#  [code]
#  @split (SEPARATOR, STRING)
#  for i to split.length
#    str$[i] = split.array$[i]
#  endfor
#
# where SEPARATOR is a separator string and STRING is a string to
# separate.
#
# If string$ = "hello world", then after
# @split (" ", string$)
# split.array$[1] contains "hello" and split.array$[2] contains "world"
#
# Notes:
# - Since .length stores the number of items separated by a string, it is always
#   larger than the amount of occurences of that string by one, which means
#   it can be used to count occurences as well.
# - This script has been changed to use the new Praat syntax, which
#   started being introduced after v.5.3.44. It can be made to work with
#   the old syntax by replacing the definition to
#
#       procedure split .sep$ .str$
#
#   and, with the same example as above, calling it with
#
#       call split " " 'string$'
#
# Written by Jose J. Atria (28 February 2012)
# Last updated: 20 February 2014
# This script is free software: you can redistribute it and/or modify 
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# A copy of the GNU General Public License is available at
# <http://www.gnu.org/licenses/>.

procedure split (.sep$, .str$)
  .seplen = length(.sep$) 
  .length = 0
  repeat
    .strlen = length(.str$)
    .sep = index(.str$, .sep$)
    if .sep > 0
      .part$ = left$(.str$, .sep-1)
      .str$ = mid$(.str$, .sep+.seplen, .strlen)
    else
      .part$ = .str$
    endif
    .length = .length+1
    .array$[.length] = .part$
  until .sep = 0
endproc