# encoding=utf-8
"""
MonPaGe Main Window Module

This module provides the abstract base class for all main windows in the MonPaGe application.
It defines common functionality and interface elements that are shared across different
application modules, such as logo display and application exit handling.

The MainWindow class serves as the foundation for module-specific windows, ensuring
consistent behavior and appearance throughout the application.

Classes:
    MainWindow: Abstract base class for module-specific main windows
"""

__author__ = "Roland Trouville"
__copyright__ = "Copyright 2015+, Consortium MonPaGe"
__license__ = "Creative Commons 4.0 By-Nc-Sa"
__maintainer__ = "Roland Trouville"
__email__ = "contact.monpage@gmail.com"
__status__ = "Production"

from typing import Optional

from PySide6.QtGui import QColor, QPalette
from PySide6.QtWidgets import QGroupBox, QLabel, QMainWindow, QWidget

from tools.display_tools import DisplayTools


class MainWindow(QMainWindow):
	"""
	Abstract base class for the main window of MonPaGe application modules.

	This class provides common functionality for all main windows in the application,
	including methods for creating logo groups and handling application exit.
	It is designed to be subclassed by specific module windows.

	Methods:
		make_logo_group: Creates a QGroupBox with MonPaGe credits and logos
		quit: Exits the application
	"""

	def __init__(self):
		super(MainWindow, self).__init__()

	@staticmethod
	def make_logo_group(parent: Optional[QWidget] = None) -> QGroupBox:
		"""
		Create and returns a QGroupBox which contains the logos and name of people. It is used in every module
		:param parent: optional parent to the resulting groupbox widget
		:returns `QGroupBox` with credits image
		"""

		group = QGroupBox("", parent)
		group.resize(DisplayTools.size(390, 570))

		logo_umons = QLabel(group)
		logo_umons.move(DisplayTools.coord(10, 10))
		logo_umons.resize(DisplayTools.size(370, 550))
		DisplayTools.display_image(logo_umons, "./ui/icons/logo/credits.png", True)

		group.setAutoFillBackground(True)
		pal = QPalette()
		color = QColor()
		color.setRgb(255, 255, 255)
		pal.setColor(QPalette.ColorRole.Window, color)
		group.setPalette(pal)

		return group

	@staticmethod
	def quit():
		"""
		Exits the application with exit code 1.
		This method terminates the program immediately, regardless of any ongoing operations.
		No cleanup operations are performed, so any unsaved data will be lost.
		"""

		exit(1)
